<?php
/**
 * Alpi Diş Hastaneleri CRM - Login Page
 * Güvenli kullanıcı girişi
 */

$public_page = true;
require_once __DIR__ . '/../includes/header.php';

// Redirect if already logged in
if (Auth::isLoggedIn()) {
    redirectTo('/public/dashboard.php');
}

$error_message = '';

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $username = sanitizeInput($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        
        // Validate CSRF token
        if (!Auth::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Güvenlik doğrulaması başarısız.');
        }
        
        // Validate inputs
        if (empty($username) || empty($password)) {
            throw new Exception('Kullanıcı adı ve şifre zorunludur.');
        }
        
        // Attempt login
        if (Auth::login($username, $password)) {
            redirectTo('/public/dashboard.php');
        }
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
    }
}

$page_title = 'Giriş';
?>

<div class="login-container">
    <div class="login-card">
        <div class="login-logo">
            <div class="logo-icon">
                <i class="fas fa-tooth"></i>
            </div>
            <h1 class="login-title">Alpi Diş Hastaneleri</h1>
            <p class="login-subtitle">CRM sistemine hoş geldiniz</p>
        </div>
        
        <?php if ($error_message): ?>
        <div class="alert alert-danger border-0" role="alert">
            <div class="d-flex align-items-center">
                <i class="fas fa-exclamation-circle me-2"></i>
                <div><?php echo htmlspecialchars($error_message); ?></div>
            </div>
        </div>
        <?php endif; ?>
        
        <form method="POST" action="" novalidate class="mt-4">
            <input type="hidden" name="csrf_token" value="<?php echo Auth::generateCSRFToken(); ?>">
            
            <div class="form-floating mb-3">
                <input type="text" class="form-control" id="username" name="username" 
                       placeholder="Kullanıcı adınızı girin"
                       value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>" 
                       required autofocus>
                <label for="username">
                    <i class="fas fa-user me-2"></i>Kullanıcı Adı
                </label>
            </div>
            
            <div class="form-floating mb-4">
                <input type="password" class="form-control" id="password" name="password" 
                       placeholder="Şifrenizi girin" required>
                <label for="password">
                    <i class="fas fa-lock me-2"></i>Şifre
                </label>
                <button class="btn btn-link position-absolute end-0 top-50 translate-middle-y me-3 p-0 border-0 bg-transparent" 
                        type="button" id="togglePassword" style="z-index: 5;">
                    <i class="fas fa-eye text-muted"></i>
                </button>
            </div>
            
            <div class="d-grid mb-4">
                <button type="submit" class="btn btn-primary btn-lg">
                    <i class="fas fa-sign-in-alt me-2"></i>
                    Sisteme Giriş Yapın
                </button>
            </div>
            
            <div class="text-center">
                <small class="text-muted">
                    <i class="fas fa-shield-alt me-1"></i>
                    Güvenli giriş
                </small>
            </div>
        </form>
        
        <div class="divider"></div>
        
        <div class="text-center">
            <small class="text-muted">
                © <?php echo date('Y'); ?> Alpi Diş Hastaneleri Grubu<br>
                Tüm hakları saklıdır.
            </small>
        </div>
    </div>
</div>

<script>
// Modern Login Interactions
document.addEventListener('DOMContentLoaded', function() {
    // Toggle password visibility
    const togglePassword = document.getElementById('togglePassword');
    const passwordField = document.getElementById('password');
    
    if (togglePassword && passwordField) {
        togglePassword.addEventListener('click', function() {
            const icon = this.querySelector('i');
            
            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
                this.setAttribute('title', 'Şifreyi gizle');
            } else {
                passwordField.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
                this.setAttribute('title', 'Şifreyi göster');
            }
        });
    }
    
    // Form validation with modern styling
    const form = document.querySelector('form');
    const usernameField = document.getElementById('username');
    
    if (form) {
        form.addEventListener('submit', function(e) {
            const username = usernameField.value.trim();
            const password = passwordField.value;
            let hasError = false;
            
            // Clear previous errors
            document.querySelectorAll('.is-invalid').forEach(el => {
                el.classList.remove('is-invalid');
            });
            
            // Validate username
            if (!username) {
                usernameField.classList.add('is-invalid');
                hasError = true;
            }
            
            // Validate password
            if (!password) {
                passwordField.classList.add('is-invalid');
                hasError = true;
            }
            
            if (hasError) {
                e.preventDefault();
                
                // Focus on first invalid field
                const firstInvalid = document.querySelector('.is-invalid');
                if (firstInvalid) {
                    firstInvalid.focus();
                }
            } else {
                // Show loading state
                const submitBtn = form.querySelector('button[type="submit"]');
                if (submitBtn) {
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Giriş yapılıyor...';
                    submitBtn.disabled = true;
                }
            }
        });
    }
    
    // Clear validation errors on input
    document.querySelectorAll('input').forEach(input => {
        input.addEventListener('input', function() {
            this.classList.remove('is-invalid');
        });
        
        input.addEventListener('focus', function() {
            this.classList.remove('is-invalid');
        });
    });
    
    // Add smooth animations
    const loginCard = document.querySelector('.login-card');
    if (loginCard) {
        loginCard.style.opacity = '0';
        loginCard.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            loginCard.style.transition = 'all 0.4s ease';
            loginCard.style.opacity = '1';
            loginCard.style.transform = 'translateY(0)';
        }, 100);
    }
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>