<?php
/**
 * Alpi Diş Hastaneleri CRM - Dashboard
 * Ana sayfa ve istatistikler
 */

require_once __DIR__ . '/../includes/header.php';

$page_title = 'Dashboard';
$breadcrumb = [];

try {
    $db = getDB();
    
    // Get statistics based on user role
    $stats = [];
    
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        // Owner can see all branches
        $sql = "SELECT 
                    COUNT(*) as total_patients,
                    COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_patients,
                    COUNT(CASE WHEN status = 'recovered' THEN 1 END) as recovered_patients,
                    COUNT(CASE WHEN status = 'lost' THEN 1 END) as lost_patients
                FROM patients 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        
    } else {
        // Other roles see only their branch
        $sql = "SELECT 
                    COUNT(*) as total_patients,
                    COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_patients,
                    COUNT(CASE WHEN status = 'recovered' THEN 1 END) as recovered_patients,
                    COUNT(CASE WHEN status = 'lost' THEN 1 END) as lost_patients
                FROM patients 
                WHERE branch_id = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        $stmt = $db->prepare($sql);
        $stmt->execute([$_SESSION['branch_id']]);
    }
    
    $stats = $stmt->fetch();
    
    // Get follow-up statistics
    $followup_stats = [];
    
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        // All branches follow-up stats
        $sql = "SELECT 
                    COUNT(*) as total_followups,
                    COUNT(CASE WHEN f.followup_date <= CURDATE() AND f.status = 'pending' THEN 1 END) as due_today,
                    COUNT(CASE WHEN f.followup_date < CURDATE() AND f.status = 'pending' THEN 1 END) as overdue,
                    COUNT(CASE WHEN f.followup_date < DATE_SUB(CURDATE(), INTERVAL 3 DAY) AND f.status = 'pending' THEN 1 END) as urgent
                FROM followups f 
                INNER JOIN patients p ON f.patient_id = p.id
                WHERE f.status IN ('pending', 'overdue')";
        $stmt = $db->prepare($sql);
        $stmt->execute();
    } else {
        // Branch specific follow-up stats
        $sql = "SELECT 
                    COUNT(*) as total_followups,
                    COUNT(CASE WHEN f.followup_date <= CURDATE() AND f.status = 'pending' THEN 1 END) as due_today,
                    COUNT(CASE WHEN f.followup_date < CURDATE() AND f.status = 'pending' THEN 1 END) as overdue,
                    COUNT(CASE WHEN f.followup_date < DATE_SUB(CURDATE(), INTERVAL 3 DAY) AND f.status = 'pending' THEN 1 END) as urgent
                FROM followups f 
                INNER JOIN patients p ON f.patient_id = p.id
                WHERE p.branch_id = ? AND f.status IN ('pending', 'overdue')";
        $stmt = $db->prepare($sql);
        $stmt->execute([$_SESSION['branch_id']]);
    }
    
    $followup_stats = $stmt->fetch();
    
    // Legacy support for existing code
    $stats['followup_count'] = $followup_stats['due_today'];
    $stats['overdue_count'] = $followup_stats['overdue'];
    $stats['urgent_count'] = $followup_stats['urgent'];
    
    // Get recent patients
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        $sql = "SELECT p.*, b.name as branch_name, c.name as consultant_name, r.reason_text
                FROM patients p
                LEFT JOIN branches b ON p.branch_id = b.id
                LEFT JOIN consultants c ON p.consultant_id = c.id
                LEFT JOIN reasons r ON p.reason_id = r.id
                ORDER BY p.created_at DESC
                LIMIT 10";
        $stmt = $db->prepare($sql);
        $stmt->execute();
    } else {
        $sql = "SELECT p.*, b.name as branch_name, c.name as consultant_name, r.reason_text
                FROM patients p
                LEFT JOIN branches b ON p.branch_id = b.id
                LEFT JOIN consultants c ON p.consultant_id = c.id
                LEFT JOIN reasons r ON p.reason_id = r.id
                WHERE p.branch_id = ?
                ORDER BY p.created_at DESC
                LIMIT 10";
        $stmt = $db->prepare($sql);
        $stmt->execute([$_SESSION['branch_id']]);
    }
    
    $recent_patients = $stmt->fetchAll();
    
    // Calculate recovery rate
    $total_completed = $stats['recovered_patients'] + $stats['lost_patients'];
    $recovery_rate = $total_completed > 0 ? round(($stats['recovered_patients'] / $total_completed) * 100, 1) : 0;
    
} catch (Exception $e) {
    showAlert('Veriler yüklenirken bir hata oluştu: ' . $e->getMessage(), 'danger');
    $stats = ['total_patients' => 0, 'pending_patients' => 0, 'recovered_patients' => 0, 'lost_patients' => 0, 'followup_count' => 0];
    $recent_patients = [];
    $recovery_rate = 0;
}
?>

<!-- Modern Page Header -->
<div class="page-header">
    <div class="d-flex justify-content-between align-items-center">
        <div>
            <h1 class="page-title">Dashboard</h1>
            <p class="page-subtitle">Hoş geldiniz, <?php echo htmlspecialchars($_SESSION['full_name']); ?>! Sisteminizin genel durumunu burada görebilirsiniz.</p>
        </div>
        <div class="d-flex gap-2">
            <div class="dropdown">
                <button class="btn btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                    <i class="fas fa-calendar"></i> Son 30 Gün
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="#"><i class="fas fa-calendar-day"></i>Bugün</a></li>
                    <li><a class="dropdown-item" href="#"><i class="fas fa-calendar-week"></i>Son 7 Gün</a></li>
                    <li><a class="dropdown-item active" href="#"><i class="fas fa-calendar-alt"></i>Son 30 Gün</a></li>
                    <li><a class="dropdown-item" href="#"><i class="fas fa-calendar"></i>Son 90 Gün</a></li>
                </ul>
            </div>
            <button type="button" class="btn btn-primary">
                <i class="fas fa-sync-alt"></i> Yenile
            </button>
        </div>
    </div>
</div>

<!-- Modern Statistics Cards -->
<div class="row g-3 mb-4">
    <div class="col-lg-3 col-md-6">
        <div class="stat-card">
            <div class="stat-icon">
                <i class="fas fa-users"></i>
            </div>
            <div class="stat-number"><?php echo number_format($stats['total_patients']); ?></div>
            <div class="stat-label">Toplam Hasta</div>
            <div class="stat-trend up">
                <i class="fas fa-arrow-up"></i> +12% bu ay
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6">
        <div class="stat-card stat-warning">
            <div class="stat-icon">
                <i class="fas fa-clock"></i>
            </div>
            <div class="stat-number"><?php echo number_format($stats['pending_patients']); ?></div>
            <div class="stat-label">Bekleyen</div>
            <div class="stat-trend">
                <i class="fas fa-minus"></i> Değişim yok
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6">
        <div class="stat-card stat-success">
            <div class="stat-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-number"><?php echo number_format($stats['recovered_patients']); ?></div>
            <div class="stat-label">Geri Kazanılan</div>
            <div class="stat-trend up">
                <i class="fas fa-arrow-up"></i> +8% bu ay
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6">
        <div class="stat-card stat-info">
            <div class="stat-icon">
                <i class="fas fa-percentage"></i>
            </div>
            <div class="stat-number"><?php echo $recovery_rate; ?>%</div>
            <div class="stat-label">Başarı Oranı</div>
            <div class="stat-trend up">
                <i class="fas fa-arrow-up"></i> +3% bu ay
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Follow-up Alerts -->
    <?php if (Auth::hasRole(Auth::ROLE_CALLCENTER) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
    <div class="col-lg-4 mb-4">
        <div class="widget">
            <div class="widget-header">
                <h3 class="widget-title">
                    <i class="fas fa-phone"></i>
                    Follow-Up Durumu
                </h3>
                <div class="widget-actions">
                    <?php if ($stats['urgent_count'] > 0): ?>
                    <span class="badge badge-danger" title="Acil"><?php echo $stats['urgent_count']; ?></span>
                    <?php endif; ?>
                    <?php if ($stats['overdue_count'] > 0): ?>
                    <span class="badge badge-warning ms-1" title="Gecikmiş"><?php echo $stats['overdue_count']; ?></span>
                    <?php endif; ?>
                </div>
            </div>
            <div class="widget-body">
                <!-- Follow-up Status Cards -->
                <div class="row g-2 mb-3">
                    <div class="col-4">
                        <div class="text-center p-2 bg-primary bg-opacity-10 rounded">
                            <div class="fw-bold text-primary"><?php echo $stats['followup_count']; ?></div>
                            <small class="text-muted">Bugün</small>
                        </div>
                    </div>
                    <div class="col-4">
                        <div class="text-center p-2 bg-warning bg-opacity-10 rounded">
                            <div class="fw-bold text-warning"><?php echo $stats['overdue_count']; ?></div>
                            <small class="text-muted">Gecikmiş</small>
                        </div>
                    </div>
                    <div class="col-4">
                        <div class="text-center p-2 bg-danger bg-opacity-10 rounded">
                            <div class="fw-bold text-danger"><?php echo $stats['urgent_count']; ?></div>
                            <small class="text-muted">Acil</small>
                        </div>
                    </div>
                </div>
                
                <?php if ($stats['followup_count'] > 0 || $stats['overdue_count'] > 0): ?>
                <!-- Alerts -->
                <?php if ($stats['urgent_count'] > 0): ?>
                <div class="alert alert-danger border-0 mb-2">
                    <div class="d-flex align-items-center">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <small><strong><?php echo $stats['urgent_count']; ?> hasta</strong> 3 gündür aranmayı bekliyor!</small>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if ($stats['overdue_count'] > 0): ?>
                <div class="alert alert-warning border-0 mb-2">
                    <div class="d-flex align-items-center">
                        <i class="fas fa-clock me-2"></i>
                        <small><strong><?php echo $stats['overdue_count']; ?> hasta</strong> aranmayı bekliyor.</small>
                    </div>
                </div>
                <?php endif; ?>
                
                <div class="d-grid">
                    <a href="/modules/followups/list.php" class="btn btn-primary btn-sm">
                        <i class="fas fa-phone"></i>
                        Follow-Up Listesi
                    </a>
                </div>
                <?php else: ?>
                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <h6 class="empty-state-title">Mükemmel!</h6>
                    <p class="mb-0">Bekleyen follow-up bulunmamaktadır.</p>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Quick Actions -->
    <div class="col-lg-4 mb-4">
        <div class="widget">
            <div class="widget-header">
                <h3 class="widget-title">
                    <i class="fas fa-bolt"></i>
                    Hızlı İşlemler
                </h3>
            </div>
            <div class="widget-body">
                <div class="d-grid gap-3">
                    <?php if (Auth::hasRole(Auth::ROLE_SALES) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
                    <a href="/modules/patients/add.php" class="btn btn-success">
                        <i class="fas fa-user-plus"></i>
                        Yeni Hasta Ekle
                    </a>
                    <?php endif; ?>
                    
                    <a href="/modules/patients/list.php" class="btn btn-outline-primary">
                        <i class="fas fa-list"></i>
                        Hasta Listesi
                    </a>
                    
                    <?php if (Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
                    <a href="/modules/reports/dashboard.php" class="btn btn-outline-secondary">
                        <i class="fas fa-chart-line"></i>
                        Raporlar
                    </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Recovery Rate Chart -->
    <div class="col-lg-4 mb-4">
        <div class="widget">
            <div class="widget-header">
                <h3 class="widget-title">
                    <i class="fas fa-chart-pie"></i>
                    Hasta Durumları
                </h3>
                <div class="widget-actions">
                    <button class="btn btn-sm btn-outline-secondary" title="Dışa Aktar">
                        <i class="fas fa-download"></i>
                    </button>
                </div>
            </div>
            <div class="widget-body">
                <div style="height: 250px;">
                    <canvas id="statusChart"></canvas>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Recent Patients -->
<div class="row">
    <div class="col-12">
        <div class="widget">
            <div class="widget-header">
                <h3 class="widget-title">
                    <i class="fas fa-clock"></i>
                    Son Eklenen Hastalar
                </h3>
                <div class="widget-actions">
                    <a href="/modules/patients/list.php" class="btn btn-sm btn-outline-primary">
                        Tümünü Gör
                    </a>
                </div>
            </div>
            <div class="widget-body p-0">
                <?php if (!empty($recent_patients)): ?>
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Hasta</th>
                                <th>Telefon</th>
                                <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                                <th>Şube</th>
                                <?php endif; ?>
                                <th>Danışman</th>
                                <th>Durum</th>
                                <th>Tarih</th>
                                <th width="120">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_patients as $patient): ?>
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="avatar-sm me-2">
                                            <div class="avatar-circle bg-light text-dark">
                                                <?php echo strtoupper(substr($patient['first_name'], 0, 1)); ?>
                                            </div>
                                        </div>
                                        <div>
                                            <div class="fw-semibold"><?php echo htmlspecialchars($patient['first_name'] . ' ' . $patient['last_name']); ?></div>
                                            <?php if ($patient['patient_type'] === 'existing'): ?>
                                            <small class="text-muted">Mevcut Hasta</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <a href="tel:<?php echo htmlspecialchars($patient['phone']); ?>" class="text-decoration-none">
                                        <?php echo htmlspecialchars($patient['phone']); ?>
                                    </a>
                                </td>
                                <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                                <td>
                                    <span class="badge badge-secondary"><?php echo htmlspecialchars($patient['branch_name']); ?></span>
                                </td>
                                <?php endif; ?>
                                <td><?php echo htmlspecialchars($patient['consultant_name'] ?? '-'); ?></td>
                                <td><?php echo getPatientStatusBadge($patient['status']); ?></td>
                                <td>
                                    <small class="text-muted"><?php echo formatDate($patient['created_at']); ?></small>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <a href="/modules/patients/view.php?id=<?php echo $patient['id']; ?>" 
                                           class="btn btn-outline-primary btn-sm" title="Görüntüle">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <?php if (Auth::hasRole(Auth::ROLE_SALES) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
                                        <a href="/modules/patients/edit.php?id=<?php echo $patient['id']; ?>" 
                                           class="btn btn-outline-secondary btn-sm" title="Düzenle">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <div class="empty-state">
                    <div class="empty-state-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <h6 class="empty-state-title">Henüz hasta yok</h6>
                    <p>İlk hasta kaydınızı oluşturmak için başlayın.</p>
                    <?php if (Auth::hasRole(Auth::ROLE_SALES) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
                    <a href="/modules/patients/add.php" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Hasta Ekle
                    </a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
// Modern Status Chart
const ctx = document.getElementById('statusChart').getContext('2d');
const statusChart = new Chart(ctx, {
    type: 'doughnut',
    data: {
        labels: ['Bekleyen', 'Geri Kazanılan', 'Kaybedilen'],
        datasets: [{
            data: [
                <?php echo $stats['pending_patients']; ?>,
                <?php echo $stats['recovered_patients']; ?>,
                <?php echo $stats['lost_patients']; ?>
            ],
            backgroundColor: [
                '#f59e0b',
                '#10b981',
                '#ef4444'
            ],
            borderWidth: 0,
            hoverBorderWidth: 2,
            hoverBorderColor: '#ffffff'
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        cutout: '60%',
        plugins: {
            legend: {
                position: 'bottom',
                labels: {
                    padding: 20,
                    usePointStyle: true,
                    pointStyle: 'circle',
                    font: {
                        size: 13,
                        family: 'Inter'
                    }
                }
            },
            tooltip: {
                backgroundColor: 'rgba(0, 0, 0, 0.8)',
                titleColor: '#ffffff',
                bodyColor: '#ffffff',
                borderColor: 'rgba(255, 255, 255, 0.1)',
                borderWidth: 1,
                cornerRadius: 8,
                displayColors: false
            }
        },
        animation: {
            animateRotate: true,
            duration: 1000
        }
    }
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>