<?php
/**
 * Alpi Diş Hastaneleri CRM - Patient List
 * Hasta listesi ve arama
 */

require_once __DIR__ . '/../../includes/header.php';

$page_title = 'Hasta Listesi';
$breadcrumb = [
    ['text' => 'Hastalar']
];

// Default values
$search = sanitizeInput($_GET['search'] ?? '');
$status_filter = $_GET['status'] ?? '';
$branch_filter = $_GET['branch'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = RECORDS_PER_PAGE;
$offset = ($page - 1) * $per_page;

try {
    $db = getDB();
    
    // Build base query
    $where_conditions = [];
    $params = [];
    
    // Branch access control
    if (!Auth::hasRole(Auth::ROLE_OWNER)) {
        $where_conditions[] = "p.branch_id = ?";
        $params[] = $_SESSION['branch_id'];
    }
    
    // Search filter
    if ($search) {
        $where_conditions[] = "(p.first_name LIKE ? OR p.last_name LIKE ? OR p.phone LIKE ?)";
        $search_param = "%$search%";
        $params[] = $search_param;
        $params[] = $search_param;
        $params[] = $search_param;
    }
    
    // Status filter
    if ($status_filter) {
        $where_conditions[] = "p.status = ?";
        $params[] = $status_filter;
    }
    
    // Branch filter (for owner)
    if ($branch_filter && Auth::hasRole(Auth::ROLE_OWNER)) {
        $where_conditions[] = "p.branch_id = ?";
        $params[] = $branch_filter;
    }
    
    $where_sql = $where_conditions ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM patients p $where_sql";
    $stmt = $db->prepare($count_sql);
    $stmt->execute($params);
    $total_records = $stmt->fetch()['total'];
    $total_pages = ceil($total_records / $per_page);
    
    // Get patients
    $sql = "SELECT p.*, b.name as branch_name, c.name as consultant_name, r.reason_text,
                   u.full_name as created_by_name
            FROM patients p
            LEFT JOIN branches b ON p.branch_id = b.id
            LEFT JOIN consultants c ON p.consultant_id = c.id
            LEFT JOIN reasons r ON p.reason_id = r.id
            LEFT JOIN users u ON p.created_by = u.id
            $where_sql
            ORDER BY p.created_at DESC
            LIMIT $per_page OFFSET $offset";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $patients = $stmt->fetchAll();
    
    // Get branches for filter (owner only)
    $branches = [];
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        $stmt = $db->prepare("SELECT id, name FROM branches WHERE is_active = 1 ORDER BY name");
        $stmt->execute();
        $branches = $stmt->fetchAll();
    }
    
    // Get statistics
    $stats_sql = "SELECT 
                    COUNT(*) as total,
                    COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending,
                    COUNT(CASE WHEN status = 'recovered' THEN 1 END) as recovered,
                    COUNT(CASE WHEN status = 'lost' THEN 1 END) as lost
                  FROM patients p";
    
    if (!Auth::hasRole(Auth::ROLE_OWNER)) {
        $stats_sql .= " WHERE p.branch_id = ?";
        $stmt = $db->prepare($stats_sql);
        $stmt->execute([$_SESSION['branch_id']]);
    } else {
        $stmt = $db->prepare($stats_sql);
        $stmt->execute();
    }
    
    $stats = $stmt->fetch();
    
} catch (Exception $e) {
    showAlert('Veriler yüklenirken bir hata oluştu: ' . $e->getMessage(), 'danger');
    $patients = [];
    $total_records = 0;
    $total_pages = 1;
    $stats = ['total' => 0, 'pending' => 0, 'recovered' => 0, 'lost' => 0];
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2">Hasta Listesi</h1>
    <div class="btn-toolbar mb-2 mb-md-0">
        <?php if (Auth::hasRole(Auth::ROLE_SALES) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
        <a href="/modules/patients/add.php" class="btn btn-primary">
            <i class="fas fa-user-plus me-2"></i>
            Yeni Hasta Ekle
        </a>
        <?php endif; ?>
    </div>
</div>

<!-- Quick Stats -->
<div class="row mb-4">
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['total']); ?></div>
                <small>Toplam Hasta</small>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-warning text-dark">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['pending']); ?></div>
                <small>Bekleyen</small>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-success text-white">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['recovered']); ?></div>
                <small>Geri Kazanılan</small>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-danger text-white">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['lost']); ?></div>
                <small>Kaybedilen</small>
            </div>
        </div>
    </div>
</div>

<!-- Search and Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-4">
                <label for="search" class="form-label">Arama</label>
                <input type="text" class="form-control" id="search" name="search" 
                       value="<?php echo htmlspecialchars($search); ?>" 
                       placeholder="Ad, soyad veya telefon...">
            </div>
            <div class="col-md-3">
                <label for="status" class="form-label">Durum</label>
                <select class="form-select" id="status" name="status">
                    <option value="">Tüm Durumlar</option>
                    <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Bekleyen</option>
                    <option value="recovered" <?php echo $status_filter === 'recovered' ? 'selected' : ''; ?>>Geri Kazanılan</option>
                    <option value="lost" <?php echo $status_filter === 'lost' ? 'selected' : ''; ?>>Kaybedilen</option>
                </select>
            </div>
            <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
            <div class="col-md-3">
                <label for="branch" class="form-label">Şube</label>
                <select class="form-select" id="branch" name="branch">
                    <option value="">Tüm Şubeler</option>
                    <?php foreach ($branches as $branch): ?>
                    <option value="<?php echo $branch['id']; ?>" 
                            <?php echo $branch_filter == $branch['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($branch['name']); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>
            <div class="col-md-2 d-flex align-items-end gap-2">
                <button type="submit" class="btn btn-outline-primary">
                    <i class="fas fa-search"></i>
                </button>
                <a href="/modules/patients/list.php" class="btn btn-outline-secondary">
                    <i class="fas fa-times"></i>
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Patients Table -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="card-title mb-0">
            <?php echo number_format($total_records); ?> hasta bulundu
        </h5>
        <?php if (!empty($patients)): ?>
        <div class="btn-group btn-group-sm">
            <button type="button" class="btn btn-outline-success" onclick="exportData('excel')">
                <i class="fas fa-file-excel me-1"></i>
                Excel
            </button>
            <button type="button" class="btn btn-outline-info" onclick="exportData('csv')">
                <i class="fas fa-file-csv me-1"></i>
                CSV
            </button>
        </div>
        <?php endif; ?>
    </div>
    <div class="card-body p-0">
        <?php if (!empty($patients)): ?>
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th>Hasta</th>
                        <th>Telefon</th>
                        <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                        <th>Şube</th>
                        <?php endif; ?>
                        <th>Danışman</th>
                        <th>Kaynak</th>
                        <th>Durum</th>
                        <th>Başvuru Tarihi</th>
                        <th>Follow-up</th>
                        <th>İşlemler</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($patients as $patient): ?>
                    <tr>
                        <td>
                            <div>
                                <strong><?php echo htmlspecialchars($patient['first_name'] . ' ' . $patient['last_name']); ?></strong>
                                <?php if ($patient['patient_type'] === 'existing'): ?>
                                <span class="badge bg-secondary ms-1">Mevcut</span>
                                <?php endif; ?>
                            </div>
                            <small class="text-muted">
                                Ekleyen: <?php echo htmlspecialchars($patient['created_by_name']); ?>
                            </small>
                        </td>
                        <td>
                            <a href="tel:<?php echo htmlspecialchars($patient['phone']); ?>" class="text-decoration-none">
                                <?php echo htmlspecialchars($patient['phone']); ?>
                            </a>
                        </td>
                        <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                        <td><?php echo htmlspecialchars($patient['branch_name']); ?></td>
                        <?php endif; ?>
                        <td><?php echo htmlspecialchars($patient['consultant_name'] ?? '-'); ?></td>
                        <td><?php echo htmlspecialchars($patient['source'] ?? '-'); ?></td>
                        <td><?php echo getPatientStatusBadge($patient['status']); ?></td>
                        <td><?php echo formatDate($patient['application_date']); ?></td>
                        <td>
                            <?php if ($patient['followup_date']): ?>
                                <?php 
                                $followup_date = strtotime($patient['followup_date']);
                                $today = strtotime('today');
                                $class = '';
                                if ($patient['status'] === 'pending') {
                                    if ($followup_date <= $today) {
                                        $class = 'text-danger fw-bold';
                                    } elseif ($followup_date <= strtotime('+2 days')) {
                                        $class = 'text-warning fw-bold';
                                    }
                                }
                                ?>
                                <span class="<?php echo $class; ?>">
                                    <?php echo formatDate($patient['followup_date']); ?>
                                </span>
                                <br>
                                <small class="text-muted">
                                    <?php echo $patient['followup_count']; ?> kez arandı
                                </small>
                            <?php else: ?>
                                <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="/modules/patients/view.php?id=<?php echo $patient['id']; ?>" 
                                   class="btn btn-outline-primary" title="Görüntüle">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <?php if (Auth::hasRole(Auth::ROLE_SALES) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
                                <a href="/modules/patients/edit.php?id=<?php echo $patient['id']; ?>" 
                                   class="btn btn-outline-secondary" title="Düzenle">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <?php endif; ?>
                                <?php if ((Auth::hasRole(Auth::ROLE_CALLCENTER) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)) && $patient['status'] === 'pending'): ?>
                                <a href="/modules/followups/update.php?patient_id=<?php echo $patient['id']; ?>" 
                                   class="btn btn-outline-success" title="Follow-up">
                                    <i class="fas fa-phone"></i>
                                </a>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
        <div class="card-footer">
            <nav aria-label="Hasta listesi sayfalama">
                <ul class="pagination pagination-sm justify-content-center mb-0">
                    <!-- Previous Page -->
                    <?php if ($page > 1): ?>
                    <li class="page-item">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                            <i class="fas fa-chevron-left"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                    
                    <!-- Page Numbers -->
                    <?php
                    $start_page = max(1, $page - 2);
                    $end_page = min($total_pages, $page + 2);
                    
                    for ($i = $start_page; $i <= $end_page; $i++):
                    ?>
                    <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                            <?php echo $i; ?>
                        </a>
                    </li>
                    <?php endfor; ?>
                    
                    <!-- Next Page -->
                    <?php if ($page < $total_pages): ?>
                    <li class="page-item">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                            <i class="fas fa-chevron-right"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
            </nav>
            
            <div class="text-center mt-2">
                <small class="text-muted">
                    Sayfa <?php echo $page; ?> / <?php echo $total_pages; ?> 
                    (Toplam <?php echo number_format($total_records); ?> kayıt)
                </small>
            </div>
        </div>
        <?php endif; ?>
        
        <?php else: ?>
        <div class="text-center py-5">
            <i class="fas fa-users fa-3x text-muted mb-3"></i>
            <h5 class="text-muted">Hasta bulunamadı</h5>
            <p class="text-muted">Arama kriterlerinizi değiştirmeyi deneyin.</p>
            <?php if (Auth::hasRole(Auth::ROLE_SALES) || Auth::hasRole(Auth::ROLE_ADMIN) || Auth::hasRole(Auth::ROLE_OWNER)): ?>
            <a href="/modules/patients/add.php" class="btn btn-primary">
                <i class="fas fa-user-plus me-2"></i>
                İlk Hastayı Ekleyin
            </a>
            <?php endif; ?>
        </div>
        <?php endif; ?>
    </div>
</div>

<script>
// Export functionality
function exportData(format) {
    const params = new URLSearchParams(window.location.search);
    params.set('export', format);
    
    // Create temporary link and trigger download
    const link = document.createElement('a');
    link.href = '/modules/patients/export.php?' + params.toString();
    link.click();
}

// Auto-refresh page every 5 minutes to update follow-up dates
setTimeout(() => {
    window.location.reload();
}, 5 * 60 * 1000);
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>