<?php
/**
 * Alpi Diş Hastaneleri CRM - Add New Patient
 * Yeni hasta ekleme formu
 */

require_once __DIR__ . '/../../includes/header.php';

// Check permissions
Auth::requireRole(Auth::ROLE_SALES);

$page_title = 'Yeni Hasta Ekle';
$breadcrumb = [
    ['text' => 'Hastalar', 'url' => '/modules/patients/list.php'],
    ['text' => 'Yeni Hasta Ekle']
];

$success_message = '';
$error_message = '';

try {
    $db = getDB();
    
    // Get branches (for owner role)
    $branches = [];
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        $stmt = $db->prepare("SELECT id, name FROM branches WHERE is_active = 1 ORDER BY name");
        $stmt->execute();
        $branches = $stmt->fetchAll();
    }
    
    // Get consultants
    $consultants = [];
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        $stmt = $db->prepare("SELECT c.*, b.name as branch_name FROM consultants c 
                            LEFT JOIN branches b ON c.branch_id = b.id 
                            WHERE c.is_active = 1 ORDER BY b.name, c.name");
        $stmt->execute();
        $consultants = $stmt->fetchAll();
    } else {
        $stmt = $db->prepare("SELECT * FROM consultants WHERE branch_id = ? AND is_active = 1 ORDER BY name");
        $stmt->execute([$_SESSION['branch_id']]);
        $consultants = $stmt->fetchAll();
    }
    
    // Get reasons
    $reasons = [];
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        $stmt = $db->prepare("SELECT * FROM reasons WHERE is_active = 1 ORDER BY reason_text");
        $stmt->execute();
        $reasons = $stmt->fetchAll();
    } else {
        $stmt = $db->prepare("SELECT * FROM reasons WHERE (branch_id = ? OR branch_id IS NULL) AND is_active = 1 ORDER BY reason_text");
        $stmt->execute([$_SESSION['branch_id']]);
        $reasons = $stmt->fetchAll();
    }
    
    // Handle form submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        
        // Validate CSRF token
        if (!Auth::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Güvenlik doğrulaması başarısız.');
        }
        
        // Collect and validate form data
        $first_name = sanitizeInput($_POST['first_name']);
        $last_name = sanitizeInput($_POST['last_name']);
        $phone = sanitizeInput($_POST['phone']);
        $branch_id = (int)($_POST['branch_id'] ?? $_SESSION['branch_id']);
        $consultant_id = !empty($_POST['consultant_id']) ? (int)$_POST['consultant_id'] : null;
        $source = sanitizeInput($_POST['source']);
        $application_date = $_POST['application_date'];
        $patient_type = $_POST['patient_type'];
        $reason_id = !empty($_POST['reason_id']) ? (int)$_POST['reason_id'] : null;
        $notes = sanitizeInput($_POST['notes']);
        
        // Validation
        if (empty($first_name) || empty($last_name) || empty($phone) || empty($application_date)) {
            throw new Exception('Ad, soyad, telefon ve başvuru tarihi zorunludur.');
        }
        
        // Validate phone number
        $phone = formatPhone($phone);
        if (!isValidPhone($phone)) {
            throw new Exception('Geçerli bir telefon numarası giriniz.');
        }
        
        // Check if patient already exists
        $stmt = $db->prepare("SELECT id FROM patients WHERE phone = ? AND branch_id = ?");
        $stmt->execute([$phone, $branch_id]);
        if ($stmt->fetch()) {
            throw new Exception('Bu telefon numarası ile bu şubede zaten bir hasta kaydı bulunmaktadır.');
        }
        
        // Validate branch access
        if (!Auth::canAccessBranch($branch_id)) {
            throw new Exception('Bu şube için yetkiniz bulunmamaktadır.');
        }
        
        // Calculate automatic follow-up date (48 hours from now)
        $followup_date = date('Y-m-d', strtotime('+2 days'));
        
        // Start transaction
        $db->beginTransaction();
        
        try {
            // Insert patient
            $stmt = $db->prepare("INSERT INTO patients 
                                (first_name, last_name, phone, branch_id, consultant_id, source, 
                                 application_date, patient_type, reason_id, notes, followup_date, 
                                 status, created_by, created_at) 
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', ?, NOW())");
            
            $stmt->execute([
                $first_name, $last_name, $phone, $branch_id, $consultant_id, $source,
                $application_date, $patient_type, $reason_id, $notes, $followup_date,
                $_SESSION['user_id']
            ]);
            
            $patient_id = $db->lastInsertId();
            
            // Create automatic follow-up record
            $stmt = $db->prepare("INSERT INTO followups 
                                (patient_id, followup_date, status, priority, auto_created, 
                                 created_by, created_at) 
                                VALUES (?, ?, 'pending', 'normal', 1, ?, NOW())");
            
            $stmt->execute([$patient_id, $followup_date, $_SESSION['user_id']]);
            
            // Log activities
            Auth::logActivity($_SESSION['user_id'], 'create', 'patients', $patient_id, 
                             "Yeni hasta eklendi: $first_name $last_name");
            
            Auth::logActivity($_SESSION['user_id'], 'create', 'followups', $db->lastInsertId(), 
                             "$first_name $last_name için otomatik follow-up oluşturuldu (Tarih: " . formatDate($followup_date) . ")");
            
            $db->commit();
            
            $success_message = "Hasta başarıyla kaydedildi. Otomatik geri arama tarihi: " . formatDate($followup_date) . " (48 saat sonra)";
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
        
        // Clear form data
        $_POST = [];
        
    }
    
} catch (Exception $e) {
    $error_message = $e->getMessage();
}

// Patient sources
$patient_sources = [
    'Google',
    'Instagram', 
    'Facebook',
    'Tavsiye',
    'Geçici Hastane',
    'Diğer'
];
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2">Yeni Hasta Ekle</h1>
    <div class="btn-toolbar mb-2 mb-md-0">
        <a href="/modules/patients/list.php" class="btn btn-outline-secondary">
            <i class="fas fa-arrow-left me-2"></i>
            Hasta Listesi
        </a>
    </div>
</div>

<?php if ($success_message): ?>
<div class="alert alert-success" role="alert">
    <i class="fas fa-check-circle me-2"></i>
    <?php echo htmlspecialchars($success_message); ?>
</div>
<?php endif; ?>

<?php if ($error_message): ?>
<div class="alert alert-danger" role="alert">
    <i class="fas fa-exclamation-triangle me-2"></i>
    <?php echo htmlspecialchars($error_message); ?>
</div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-user-plus me-2"></i>
                    Hasta Bilgileri
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" data-validate>
                    <input type="hidden" name="csrf_token" value="<?php echo Auth::generateCSRFToken(); ?>">
                    
                    <!-- Personal Information -->
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="first_name" class="form-label">Ad *</label>
                            <input type="text" class="form-control" id="first_name" name="first_name" 
                                   value="<?php echo htmlspecialchars($_POST['first_name'] ?? ''); ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="last_name" class="form-label">Soyad *</label>
                            <input type="text" class="form-control" id="last_name" name="last_name" 
                                   value="<?php echo htmlspecialchars($_POST['last_name'] ?? ''); ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="phone" class="form-label">Telefon *</label>
                            <input type="tel" class="form-control" id="phone" name="phone" 
                                   value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>" 
                                   placeholder="05XX XXX XX XX" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="patient_type" class="form-label">Hasta Tipi</label>
                            <select class="form-select" id="patient_type" name="patient_type">
                                <option value="new" <?php echo ($_POST['patient_type'] ?? 'new') === 'new' ? 'selected' : ''; ?>>Yeni Hasta</option>
                                <option value="existing" <?php echo ($_POST['patient_type'] ?? '') === 'existing' ? 'selected' : ''; ?>>Mevcut Hasta</option>
                            </select>
                        </div>
                    </div>
                    
                    <!-- Branch and Consultant -->
                    <div class="row">
                        <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                        <div class="col-md-6 mb-3">
                            <label for="branch_id" class="form-label">Şube *</label>
                            <select class="form-select" id="branch_id" name="branch_id" required>
                                <option value="">Şube Seçiniz</option>
                                <?php foreach ($branches as $branch): ?>
                                <option value="<?php echo $branch['id']; ?>" 
                                        <?php echo ($_POST['branch_id'] ?? '') == $branch['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($branch['name']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php else: ?>
                        <input type="hidden" name="branch_id" value="<?php echo $_SESSION['branch_id']; ?>">
                        <?php endif; ?>
                        
                        <div class="col-md-6 mb-3">
                            <label for="consultant_id" class="form-label">Hasta Danışmanı</label>
                            <select class="form-select" id="consultant_id" name="consultant_id">
                                <option value="">Danışman Seçiniz</option>
                                <?php foreach ($consultants as $consultant): ?>
                                <option value="<?php echo $consultant['id']; ?>" 
                                        data-branch="<?php echo $consultant['branch_id']; ?>"
                                        <?php echo ($_POST['consultant_id'] ?? '') == $consultant['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($consultant['name']); ?>
                                    <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                                        (<?php echo htmlspecialchars($consultant['branch_name']); ?>)
                                    <?php endif; ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <!-- Source and Date -->
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="source" class="form-label">Bize Nereden Eriştiniz?</label>
                            <select class="form-select" id="source" name="source">
                                <option value="">Kaynak Seçiniz</option>
                                <?php foreach ($patient_sources as $source): ?>
                                <option value="<?php echo $source; ?>" 
                                        <?php echo ($_POST['source'] ?? '') === $source ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($source); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="application_date" class="form-label">Başvuru Tarihi *</label>
                            <input type="date" class="form-control" id="application_date" name="application_date" 
                                   value="<?php echo $_POST['application_date'] ?? date('Y-m-d'); ?>" 
                                   max="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                    </div>
                    
                    <!-- Reason -->
                    <div class="mb-3">
                        <label for="reason_id" class="form-label">Neden Anlaşma Yapılmadı?</label>
                        <select class="form-select" id="reason_id" name="reason_id">
                            <option value="">Neden Seçiniz</option>
                            <?php foreach ($reasons as $reason): ?>
                            <option value="<?php echo $reason['id']; ?>" 
                                    <?php echo ($_POST['reason_id'] ?? '') == $reason['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($reason['reason_text']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Notes -->
                    <div class="mb-4">
                        <label for="notes" class="form-label">Ek Notlar</label>
                        <textarea class="form-control" id="notes" name="notes" rows="3" 
                                  placeholder="Hasta ile ilgili ek bilgiler..."><?php echo htmlspecialchars($_POST['notes'] ?? ''); ?></textarea>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>
                            Hasta Kaydet
                        </button>
                        <button type="reset" class="btn btn-outline-secondary">
                            <i class="fas fa-undo me-2"></i>
                            Formu Temizle
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Help Card -->
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-info-circle me-2"></i>
                    Bilgilendirme
                </h6>
            </div>
            <div class="card-body">
                <div class="alert alert-info">
                    <small>
                        <strong>Geri Arama:</strong><br>
                        Hasta kaydedildikten sonra <?php echo DEFAULT_FOLLOWUP_DAYS; ?> gün sonra 
                        otomatik olarak çağrı merkezine geri arama bildirimi gönderilecektir.
                    </small>
                </div>
                
                <div class="alert alert-warning">
                    <small>
                        <strong>Telefon Formatı:</strong><br>
                        Telefon numarası 0 ile başlamalı ve 11 haneli olmalıdır.
                        Örnek: 05XX XXX XX XX
                    </small>
                </div>
                
                <div class="small text-muted">
                    <strong>Zorunlu Alanlar:</strong>
                    <ul class="mb-0 mt-2">
                        <li>Ad</li>
                        <li>Soyad</li>
                        <li>Telefon</li>
                        <li>Başvuru Tarihi</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Filter consultants by branch
<?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
document.getElementById('branch_id').addEventListener('change', function() {
    const branchId = this.value;
    const consultantSelect = document.getElementById('consultant_id');
    const consultantOptions = consultantSelect.querySelectorAll('option[data-branch]');
    
    // Show/hide consultant options based on selected branch
    consultantOptions.forEach(option => {
        if (!branchId || option.dataset.branch === branchId) {
            option.style.display = 'block';
        } else {
            option.style.display = 'none';
        }
    });
    
    // Reset consultant selection if not valid for selected branch
    if (branchId && consultantSelect.value) {
        const selectedOption = consultantSelect.querySelector(`option[value="${consultantSelect.value}"]`);
        if (selectedOption && selectedOption.dataset.branch !== branchId) {
            consultantSelect.value = '';
        }
    }
});

// Trigger filter on page load
document.getElementById('branch_id').dispatchEvent(new Event('change'));
<?php endif; ?>

// Phone number formatting (handled by app.js)
// Form validation (handled by app.js)
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>