<?php
/**
 * Alpi Diş Hastaneleri CRM - Follow-up List
 * Geri arama listesi ve takip
 */

require_once __DIR__ . '/../../includes/header.php';

// Check permissions
Auth::requireRole(Auth::ROLE_CALLCENTER);

$page_title = 'Follow-Up Listesi';
$breadcrumb = [
    ['text' => 'Follow-Up']
];

// Default values
$status_filter = $_GET['status'] ?? 'due';
$branch_filter = $_GET['branch'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = RECORDS_PER_PAGE;
$offset = ($page - 1) * $per_page;

try {
    $db = getDB();
    
    // Build base query
    $where_conditions = ["p.status = 'pending'"];
    $params = [];
    
    // Branch access control
    if (!Auth::hasRole(Auth::ROLE_OWNER)) {
        $where_conditions[] = "p.branch_id = ?";
        $params[] = $_SESSION['branch_id'];
    }
    
    // Status filter
    switch ($status_filter) {
        case 'due':
            $where_conditions[] = "p.followup_date <= CURDATE()";
            break;
        case 'today':
            $where_conditions[] = "p.followup_date = CURDATE()";
            break;
        case 'overdue':
            $where_conditions[] = "p.followup_date < CURDATE()";
            break;
        case 'upcoming':
            $where_conditions[] = "p.followup_date > CURDATE()";
            break;
    }
    
    // Branch filter (for owner)
    if ($branch_filter && Auth::hasRole(Auth::ROLE_OWNER)) {
        $where_conditions[] = "p.branch_id = ?";
        $params[] = $branch_filter;
    }
    
    $where_sql = 'WHERE ' . implode(' AND ', $where_conditions);
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM patients p $where_sql";
    $stmt = $db->prepare($count_sql);
    $stmt->execute($params);
    $total_records = $stmt->fetch()['total'];
    $total_pages = ceil($total_records / $per_page);
    
    // Get patients for follow-up
    $sql = "SELECT p.*, b.name as branch_name, c.name as consultant_name, r.reason_text,
                   u.full_name as created_by_name,
                   DATEDIFF(CURDATE(), p.followup_date) as days_overdue
            FROM patients p
            LEFT JOIN branches b ON p.branch_id = b.id
            LEFT JOIN consultants c ON p.consultant_id = c.id
            LEFT JOIN reasons r ON p.reason_id = r.id
            LEFT JOIN users u ON p.created_by = u.id
            $where_sql
            ORDER BY p.followup_date ASC, p.created_at ASC
            LIMIT $per_page OFFSET $offset";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $patients = $stmt->fetchAll();
    
    // Get branches for filter (owner only)
    $branches = [];
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        $stmt = $db->prepare("SELECT id, name FROM branches WHERE is_active = 1 ORDER BY name");
        $stmt->execute();
        $branches = $stmt->fetchAll();
    }
    
    // Get follow-up statistics
    $stats_sql = "SELECT 
                    COUNT(CASE WHEN followup_date = CURDATE() THEN 1 END) as today,
                    COUNT(CASE WHEN followup_date < CURDATE() THEN 1 END) as overdue,
                    COUNT(CASE WHEN followup_date <= CURDATE() THEN 1 END) as due,
                    COUNT(CASE WHEN followup_date > CURDATE() THEN 1 END) as upcoming
                  FROM patients p WHERE status = 'pending'";
    
    if (!Auth::hasRole(Auth::ROLE_OWNER)) {
        $stats_sql .= " AND p.branch_id = ?";
        $stmt = $db->prepare($stats_sql);
        $stmt->execute([$_SESSION['branch_id']]);
    } else {
        $stmt = $db->prepare($stats_sql);
        $stmt->execute();
    }
    
    $stats = $stmt->fetch();
    
} catch (Exception $e) {
    showAlert('Veriler yüklenirken bir hata oluştu: ' . $e->getMessage(), 'danger');
    $patients = [];
    $total_records = 0;
    $total_pages = 1;
    $stats = ['today' => 0, 'overdue' => 0, 'due' => 0, 'upcoming' => 0];
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2">Follow-Up Listesi</h1>
    <div class="btn-toolbar mb-2 mb-md-0">
        <div class="btn-group me-2">
            <button type="button" class="btn btn-outline-info btn-sm" onclick="refreshPage()">
                <i class="fas fa-refresh me-2"></i>
                Yenile
            </button>
        </div>
    </div>
</div>

<!-- Quick Stats -->
<div class="row mb-4">
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-danger text-white">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['overdue']); ?></div>
                <small>Gecikmiş</small>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-warning text-dark">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['today']); ?></div>
                <small>Bugün</small>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-primary text-white">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['due']); ?></div>
                <small>Aranacak</small>
            </div>
        </div>
    </div>
    <div class="col-lg-3 col-md-6 mb-2">
        <div class="card bg-info text-white">
            <div class="card-body text-center">
                <div class="h3 mb-0"><?php echo number_format($stats['upcoming']); ?></div>
                <small>Gelecek</small>
            </div>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-4">
                <label for="status" class="form-label">Durum</label>
                <select class="form-select" id="status" name="status">
                    <option value="due" <?php echo $status_filter === 'due' ? 'selected' : ''; ?>>Aranacaklar</option>
                    <option value="today" <?php echo $status_filter === 'today' ? 'selected' : ''; ?>>Bugün</option>
                    <option value="overdue" <?php echo $status_filter === 'overdue' ? 'selected' : ''; ?>>Gecikmiş</option>
                    <option value="upcoming" <?php echo $status_filter === 'upcoming' ? 'selected' : ''; ?>>Gelecek</option>
                </select>
            </div>
            <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
            <div class="col-md-4">
                <label for="branch" class="form-label">Şube</label>
                <select class="form-select" id="branch" name="branch">
                    <option value="">Tüm Şubeler</option>
                    <?php foreach ($branches as $branch): ?>
                    <option value="<?php echo $branch['id']; ?>" 
                            <?php echo $branch_filter == $branch['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($branch['name']); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>
            <div class="col-md-2 d-flex align-items-end">
                <button type="submit" class="btn btn-outline-primary">
                    <i class="fas fa-filter me-2"></i>
                    Filtrele
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Follow-up Table -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="card-title mb-0">
            <?php echo number_format($total_records); ?> hasta follow-up bekliyor
        </h5>
    </div>
    <div class="card-body p-0">
        <?php if (!empty($patients)): ?>
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th>Öncelik</th>
                        <th>Hasta</th>
                        <th>Telefon</th>
                        <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                        <th>Şube</th>
                        <?php endif; ?>
                        <th>Danışman</th>
                        <th>Neden</th>
                        <th>Follow-up Tarihi</th>
                        <th>Arama Sayısı</th>
                        <th>İşlemler</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($patients as $patient): ?>
                    <tr <?php echo $patient['days_overdue'] > 0 ? 'class="table-danger"' : ($patient['days_overdue'] == 0 ? 'class="table-warning"' : ''); ?>>
                        <td>
                            <?php if ($patient['days_overdue'] > 0): ?>
                                <span class="badge bg-danger">
                                    <i class="fas fa-exclamation-triangle"></i>
                                    <?php echo $patient['days_overdue']; ?> gün gecikmiş
                                </span>
                            <?php elseif ($patient['days_overdue'] == 0): ?>
                                <span class="badge bg-warning text-dark">
                                    <i class="fas fa-clock"></i>
                                    Bugün
                                </span>
                            <?php else: ?>
                                <span class="badge bg-info">
                                    <i class="fas fa-calendar"></i>
                                    <?php echo abs($patient['days_overdue']); ?> gün sonra
                                </span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div>
                                <strong><?php echo htmlspecialchars($patient['first_name'] . ' ' . $patient['last_name']); ?></strong>
                                <?php if ($patient['patient_type'] === 'existing'): ?>
                                <span class="badge bg-secondary ms-1">Mevcut</span>
                                <?php endif; ?>
                            </div>
                            <small class="text-muted">
                                Ekleyen: <?php echo htmlspecialchars($patient['created_by_name']); ?>
                            </small>
                        </td>
                        <td>
                            <a href="tel:<?php echo htmlspecialchars($patient['phone']); ?>" 
                               class="btn btn-outline-success btn-sm">
                                <i class="fas fa-phone me-1"></i>
                                <?php echo htmlspecialchars($patient['phone']); ?>
                            </a>
                        </td>
                        <?php if (Auth::hasRole(Auth::ROLE_OWNER)): ?>
                        <td><?php echo htmlspecialchars($patient['branch_name']); ?></td>
                        <?php endif; ?>
                        <td><?php echo htmlspecialchars($patient['consultant_name'] ?? '-'); ?></td>
                        <td>
                            <?php if ($patient['reason_text']): ?>
                                <span class="badge bg-light text-dark">
                                    <?php echo htmlspecialchars($patient['reason_text']); ?>
                                </span>
                            <?php else: ?>
                                <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <strong><?php echo formatDate($patient['followup_date']); ?></strong>
                            <br>
                            <small class="text-muted">
                                Başvuru: <?php echo formatDate($patient['application_date']); ?>
                            </small>
                        </td>
                        <td>
                            <span class="badge <?php echo $patient['followup_count'] >= MAX_FOLLOWUP_ATTEMPTS ? 'bg-danger' : 'bg-primary'; ?>">
                                <?php echo $patient['followup_count']; ?> / <?php echo MAX_FOLLOWUP_ATTEMPTS; ?>
                            </span>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="/modules/patients/view.php?id=<?php echo $patient['id']; ?>" 
                                   class="btn btn-outline-info" title="Detay">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <a href="/modules/followups/update.php?patient_id=<?php echo $patient['id']; ?>" 
                                   class="btn btn-primary" title="Ara">
                                    <i class="fas fa-phone"></i>
                                    Ara
                                </a>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
        <div class="card-footer">
            <nav aria-label="Follow-up listesi sayfalama">
                <ul class="pagination pagination-sm justify-content-center mb-0">
                    <!-- Previous Page -->
                    <?php if ($page > 1): ?>
                    <li class="page-item">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                            <i class="fas fa-chevron-left"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                    
                    <!-- Page Numbers -->
                    <?php
                    $start_page = max(1, $page - 2);
                    $end_page = min($total_pages, $page + 2);
                    
                    for ($i = $start_page; $i <= $end_page; $i++):
                    ?>
                    <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                            <?php echo $i; ?>
                        </a>
                    </li>
                    <?php endfor; ?>
                    
                    <!-- Next Page -->
                    <?php if ($page < $total_pages): ?>
                    <li class="page-item">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                            <i class="fas fa-chevron-right"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
            </nav>
            
            <div class="text-center mt-2">
                <small class="text-muted">
                    Sayfa <?php echo $page; ?> / <?php echo $total_pages; ?> 
                    (Toplam <?php echo number_format($total_records); ?> kayıt)
                </small>
            </div>
        </div>
        <?php endif; ?>
        
        <?php else: ?>
        <div class="text-center py-5">
            <i class="fas fa-phone fa-3x text-muted mb-3"></i>
            <h5 class="text-muted">Follow-up bekleyen hasta bulunamadı</h5>
            <p class="text-muted">Seçilen kriterlere göre aranacak hasta bulunmamaktadır.</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Help Modal -->
<div class="modal fade" id="helpModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Follow-Up Yardım</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <h6>Öncelik Renkleri:</h6>
                <ul class="list-unstyled">
                    <li><span class="badge bg-danger me-2">Kırmızı</span> Gecikmiş aramalar</li>
                    <li><span class="badge bg-warning text-dark me-2">Sarı</span> Bugün aranacaklar</li>
                    <li><span class="badge bg-info me-2">Mavi</span> Gelecek aramalar</li>
                </ul>
                
                <h6 class="mt-3">Arama Sınırı:</h6>
                <p>Her hasta maksimum <?php echo MAX_FOLLOWUP_ATTEMPTS; ?> kez aranabilir.</p>
                
                <h6 class="mt-3">Durum Güncellemeleri:</h6>
                <ul>
                    <li><strong>Geri Kazanıldı:</strong> Hasta tedaviyi kabul etti</li>
                    <li><strong>Kaybedildi:</strong> Hasta kesin red verdi</li>
                    <li><strong>Tekrar Ara:</strong> Daha sonra aranacak</li>
                </ul>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Kapat</button>
            </div>
        </div>
    </div>
</div>

<script>
// Refresh page
function refreshPage() {
    window.location.reload();
}

// Auto-refresh every 2 minutes
setInterval(refreshPage, 2 * 60 * 1000);

// Show help modal on first visit
if (!localStorage.getItem('followup_help_shown')) {
    setTimeout(() => {
        const helpModal = new bootstrap.Modal(document.getElementById('helpModal'));
        helpModal.show();
        localStorage.setItem('followup_help_shown', 'true');
    }, 1000);
}
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>